// src/ast.ts
import { walkIdentifiers } from "@vue/compiler-sfc";
import { isFunctionType, isLiteralType } from "ast-kit";
function checkInvalidScopeReference(node, method, setupBindings) {
  if (!node)
    return;
  walkIdentifiers(node, (id) => {
    if (setupBindings.includes(id.name))
      throw new SyntaxError(
        `\`${method}()\` in <script setup> cannot reference locally declared variables (${id.name}) because it will be hoisted outside of the setup() function.`
      );
  });
}
function isStaticExpression(node, options = {}) {
  const { magicComment, fn, object, objectMethod, array, unary, regex } = options;
  if (magicComment && node.leadingComments?.some(
    (comment) => comment.value.trim() === magicComment
  ))
    return true;
  else if (fn && isFunctionType(node))
    return true;
  switch (node.type) {
    case "UnaryExpression":
      return !!unary && isStaticExpression(node.argument, options);
    case "LogicalExpression":
    case "BinaryExpression":
      return isStaticExpression(node.left, options) && isStaticExpression(node.right, options);
    case "ConditionalExpression":
      return isStaticExpression(node.test, options) && isStaticExpression(node.consequent, options) && isStaticExpression(node.alternate, options);
    case "SequenceExpression":
    case "TemplateLiteral":
      return node.expressions.every((expr) => isStaticExpression(expr, options));
    case "ArrayExpression":
      return !!array && node.elements.every(
        (element) => element && isStaticExpression(element, options)
      );
    case "ObjectExpression":
      return !!object && node.properties.every((prop) => {
        if (prop.type === "SpreadElement") {
          return prop.argument.type === "ObjectExpression" && isStaticExpression(prop.argument, options);
        } else if (!isLiteralType(prop.key) && prop.computed) {
          return false;
        } else if (prop.type === "ObjectProperty" && !isStaticExpression(prop.value, options)) {
          return false;
        }
        if (prop.type === "ObjectMethod" && !objectMethod) {
          return false;
        }
        return true;
      });
    case "ParenthesizedExpression":
    case "TSNonNullExpression":
    case "TSAsExpression":
    case "TSTypeAssertion":
    case "TSSatisfiesExpression":
      return isStaticExpression(node.expression, options);
    case "RegExpLiteral":
      return !!regex;
  }
  if (isLiteralType(node))
    return true;
  return false;
}
function isStaticObjectKey(node) {
  return node.properties.every((prop) => {
    if (prop.type === "SpreadElement") {
      return prop.argument.type === "ObjectExpression" && isStaticObjectKey(prop.argument);
    }
    return !prop.computed || isLiteralType(prop.key);
  });
}
function resolveObjectExpression(node) {
  const maps = {};
  for (const property of node.properties) {
    if (property.type === "SpreadElement") {
      if (property.argument.type !== "ObjectExpression")
        return void 0;
      Object.assign(maps, resolveObjectExpression(property.argument));
    } else {
      const key = resolveObjectKey(property.key, property.computed, false);
      maps[key] = property;
    }
  }
  return maps;
}
function resolveObjectKey(node, computed = false, raw = true) {
  switch (node.type) {
    case "StringLiteral":
    case "NumericLiteral":
      return raw ? node.extra.raw : node.value;
    case "Identifier":
      if (!computed)
        return raw ? `'${node.name}'` : node.name;
    default:
      throw new SyntaxError(`Unexpected node type: ${node.type}`);
  }
}
var importedMap = /* @__PURE__ */ new WeakMap();
var HELPER_PREFIX = "__MACROS_";
function importHelperFn(s, offset, local, from = "vue", isDefault = false) {
  const imported = isDefault ? "default" : local;
  const cacheKey = `${from}@${imported}`;
  if (!importedMap.get(s)?.has(cacheKey)) {
    s.appendLeft(
      offset,
      `
import ${isDefault ? HELPER_PREFIX + local : `{ ${imported} as ${HELPER_PREFIX + local} }`} from ${JSON.stringify(from)};`
    );
    if (!importedMap.has(s)) {
      importedMap.set(s, /* @__PURE__ */ new Set([cacheKey]));
    } else {
      importedMap.get(s).add(cacheKey);
    }
  }
  return `${HELPER_PREFIX}${local}`;
}

export {
  checkInvalidScopeReference,
  isStaticExpression,
  isStaticObjectKey,
  resolveObjectExpression,
  resolveObjectKey,
  HELPER_PREFIX,
  importHelperFn
};
