'use strict';

const vue = require('vue');
const unhead = require('unhead');
const shared = require('@unhead/shared');

function resolveUnref(r) {
  return typeof r === "function" ? r() : vue.unref(r);
}
function resolveUnrefHeadInput(ref, lastKey = "") {
  if (ref instanceof Promise)
    return ref;
  const root = resolveUnref(ref);
  if (!ref || !root)
    return root;
  if (Array.isArray(root))
    return root.map((r) => resolveUnrefHeadInput(r, lastKey));
  if (typeof root === "object") {
    return Object.fromEntries(
      Object.entries(root).map(([k, v]) => {
        if (k === "titleTemplate" || k.startsWith("on"))
          return [k, vue.unref(v)];
        return [k, resolveUnrefHeadInput(v, k)];
      })
    );
  }
  return root;
}

const Vue3 = vue.version.startsWith("3");
const IsBrowser = typeof window !== "undefined";

const headSymbol = "usehead";
function injectHead() {
  return vue.getCurrentInstance() && vue.inject(headSymbol) || unhead.getActiveHead();
}
function vueInstall(head) {
  const plugin = {
    install(app) {
      if (Vue3) {
        app.config.globalProperties.$unhead = head;
        app.config.globalProperties.$head = head;
        app.provide(headSymbol, head);
      }
    }
  };
  return plugin.install;
}
function createServerHead(options = {}) {
  const head = unhead.createServerHead({
    ...options,
    plugins: [
      VueReactiveUseHeadPlugin(),
      ...options?.plugins || []
    ]
  });
  head.install = vueInstall(head);
  return head;
}
function createHead(options = {}) {
  const head = unhead.createHead({
    ...options,
    // arbitrary delay the dom update for batch updates
    domDelayFn: (fn) => setTimeout(() => vue.nextTick(() => fn()), 10),
    plugins: [
      VueReactiveUseHeadPlugin(),
      ...options?.plugins || []
    ]
  });
  head.install = vueInstall(head);
  return head;
}

function VueReactiveUseHeadPlugin() {
  return shared.defineHeadPlugin({
    hooks: {
      "entries:resolve": function(ctx) {
        for (const entry of ctx.entries)
          entry.resolvedInput = resolveUnrefHeadInput(entry.input);
      }
    }
  });
}

function clientUseHead(input, options = {}) {
  const head = injectHead();
  const deactivated = vue.ref(false);
  const resolvedInput = vue.ref({});
  vue.watchEffect(() => {
    resolvedInput.value = deactivated.value ? {} : resolveUnrefHeadInput(input);
  });
  const entry = head.push(resolvedInput.value, options);
  vue.watch(resolvedInput, (e) => {
    entry.patch(e);
  });
  const vm = vue.getCurrentInstance();
  if (vm) {
    vue.onBeforeUnmount(() => {
      entry.dispose();
    });
    vue.onDeactivated(() => {
      deactivated.value = true;
    });
    vue.onActivated(() => {
      deactivated.value = false;
    });
  }
  return entry;
}

function serverUseHead(input, options = {}) {
  const head = injectHead();
  return head.push(input, options);
}

function useHead(input, options = {}) {
  const head = injectHead();
  if (head) {
    const isBrowser = IsBrowser || !!head.resolvedOptions?.document;
    if (options.mode === "server" && isBrowser || options.mode === "client" && !isBrowser)
      return;
    return isBrowser ? clientUseHead(input, options) : serverUseHead(input, options);
  }
}

exports.VueReactiveUseHeadPlugin = VueReactiveUseHeadPlugin;
exports.createHead = createHead;
exports.createServerHead = createServerHead;
exports.headSymbol = headSymbol;
exports.injectHead = injectHead;
exports.resolveUnrefHeadInput = resolveUnrefHeadInput;
exports.serverUseHead = serverUseHead;
exports.useHead = useHead;
