'use strict';

function asArray(value) {
  return Array.isArray(value) ? value : [value];
}

const SelfClosingTags = ["meta", "link", "base"];
const TagsWithInnerContent = ["title", "script", "style", "noscript"];
const HasElementTags = [
  "base",
  "meta",
  "link",
  "style",
  "script",
  "noscript"
];
const ValidHeadTags = [
  "title",
  "titleTemplate",
  "templateParams",
  "base",
  "htmlAttrs",
  "bodyAttrs",
  "meta",
  "link",
  "style",
  "script",
  "noscript"
];
const UniqueTags = ["base", "title", "titleTemplate", "bodyAttrs", "htmlAttrs", "templateParams"];
const TagConfigKeys = ["tagPosition", "tagPriority", "tagDuplicateStrategy", "innerHTML", "textContent"];

function defineHeadPlugin(plugin) {
  return plugin;
}

function hashCode(s) {
  let h = 9;
  for (let i = 0; i < s.length; )
    h = Math.imul(h ^ s.charCodeAt(i++), 9 ** 9);
  return ((h ^ h >>> 9) + 65536).toString(16).substring(1, 8).toLowerCase();
}
function hashTag(tag) {
  return hashCode(`${tag.tag}:${tag.textContent || tag.innerHTML || ""}:${Object.entries(tag.props).map(([key, value]) => `${key}:${String(value)}`).join(",")}`);
}
function computeHashes(hashes) {
  let h = 9;
  for (const s of hashes) {
    for (let i = 0; i < s.length; )
      h = Math.imul(h ^ s.charCodeAt(i++), 9 ** 9);
  }
  return ((h ^ h >>> 9) + 65536).toString(16).substring(1, 8).toLowerCase();
}

function tagDedupeKey(tag, fn) {
  const { props, tag: tagName } = tag;
  if (UniqueTags.includes(tagName))
    return tagName;
  if (tagName === "link" && props.rel === "canonical")
    return "canonical";
  if (props.charset)
    return "charset";
  const name = ["id"];
  if (tagName === "meta")
    name.push(...["name", "property", "http-equiv"]);
  for (const n of name) {
    if (typeof props[n] !== "undefined") {
      const val = String(props[n]);
      if (fn && !fn(val))
        return false;
      return `${tagName}:${n}:${val}`;
    }
  }
  return false;
}

function resolveTitleTemplate(template, title) {
  if (template == null)
    return title || null;
  if (typeof template === "function")
    return template(title);
  return template;
}

exports.HasElementTags = HasElementTags;
exports.SelfClosingTags = SelfClosingTags;
exports.TagConfigKeys = TagConfigKeys;
exports.TagsWithInnerContent = TagsWithInnerContent;
exports.UniqueTags = UniqueTags;
exports.ValidHeadTags = ValidHeadTags;
exports.asArray = asArray;
exports.computeHashes = computeHashes;
exports.defineHeadPlugin = defineHeadPlugin;
exports.hashCode = hashCode;
exports.hashTag = hashTag;
exports.resolveTitleTemplate = resolveTitleTemplate;
exports.tagDedupeKey = tagDedupeKey;
